/*
 *  ======== edma.c ========
 *  This is the code needed to setup the edma.
 */

/**************************************************************/

/*
 *  ======== Include files ========
 */
#include <csl.h>
#include <csl_edma.h>


#include "mcbsp.h"
#include "dsk6416.h"
#include "dsk6416_dip.h"

/*
 *  ======== Declarations ========
 */
#define BUFFSIZE 512
#define PING 0
#define PONG 1

/*
 *  ======== Prototypes ========
 */
void initEdma(void);
void edmaHwi(int tcc);

/*
 *  ======== References ========
 */
extern short gBufRcvLtPing[BUFFSIZE];
extern short gBufRcvLtPong[BUFFSIZE];
extern short gBufRcvRtPing[BUFFSIZE];
extern short gBufRcvRtPong[BUFFSIZE];
extern short gBufXmtLtPing[BUFFSIZE];
extern short gBufXmtLtPong[BUFFSIZE];
extern short gBufXmtRtPing[BUFFSIZE];
extern short gBufXmtRtPong[BUFFSIZE];

/*
 *  ======== Global Variables ========
 */
EDMA_Handle hEdmaRcv;
EDMA_Handle hEdmaReloadRcvPing;
EDMA_Handle hEdmaReloadRcvPong;

EDMA_Handle hEdmaXmt;
EDMA_Handle hEdmaReloadXmtPing;
EDMA_Handle hEdmaReloadXmtPong;

short gXmtTCC;
short gRcvTCC;

EDMA_Config gEdmaConfigRcv = {  
	EDMA_OPT_RMK(
		EDMA_OPT_PRI_LOW,		 // Priority                                     
		EDMA_OPT_ESIZE_16BIT,	 // Element size                                 
		EDMA_OPT_2DS_NO,		 // 2 dimensional source                         
		EDMA_OPT_SUM_NONE,		 // Src update mode                              
		EDMA_OPT_2DD_NO,		 // 2 dimensional dest                           
		EDMA_OPT_DUM_IDX,		 // Dest update mode                             
		EDMA_OPT_TCINT_YES,		 // Cause EDMA interrupt                         
		EDMA_OPT_TCC_OF(0),		 // Transfer Complete Code                       
		EDMA_OPT_TCCM_DEFAULT,	 // Transfer Complete Code Upper Bits (c64x only)
		EDMA_OPT_ATCINT_DEFAULT, // Alternate TCC Interrupt (c64x only)          
		EDMA_OPT_ATCC_DEFAULT,	 // Alternate Transfer Complete Code (c64x only) 
		EDMA_OPT_PDTS_DEFAULT,	 // Peripheral Device Transfer Source (c64x only)
		EDMA_OPT_PDTD_DEFAULT,	 // Peripheral Device Transfer Dest (c64x only)  
		EDMA_OPT_LINK_YES,		 // Enable link parameters                       
		EDMA_OPT_FS_NO			 // Use frame sync                               
	),							                                                 
	EDMA_SRC_OF(0),							 // src address                                  
	EDMA_CNT_RMK(							 // Framecount/Elementcount
		EDMA_CNT_FRMCNT_OF(BUFFSIZE - 1),	 // Framecount = # of buffers -1
		EDMA_CNT_ELECNT_OF(2)				 // Elementcount = number of elements per frame
	),
	EDMA_DST_OF(gBufRcvLtPing),					 // dest address
	EDMA_IDX_RMK(							 // Frame Index/Element Index
		EDMA_IDX_FRMIDX_OF(-(BUFFSIZE*2)+2), // Negative Frame Index to move us back to the previous channel
		EDMA_IDX_ELEIDX_OF(BUFFSIZE*2)		 // Positive Element Index to move us to the next channel
	),
	EDMA_RLD_RMK(							 // Element Reload needed when doing a synchronized, frame indexed transfer
		EDMA_RLD_ELERLD_OF(2),				 // Number of elements, should be the same as Element Count
		EDMA_RLD_LINK_OF(0)					 // Done in code with EDMA_link()
	)
};

EDMA_Config gEdmaConfigXmt = {  
	EDMA_OPT_RMK(
		EDMA_OPT_PRI_LOW,		 // Priority                                     
		EDMA_OPT_ESIZE_16BIT,	 // Element size                                 
		EDMA_OPT_2DS_NO,		 // 2 dimensional source                         
		EDMA_OPT_SUM_IDX,		 // Src update mode                              
		EDMA_OPT_2DD_NO,		 // 2 dimensional dest                           
		EDMA_OPT_DUM_NONE,		 // Dest update mode                             
		EDMA_OPT_TCINT_YES,		 // Cause EDMA interrupt                         
		EDMA_OPT_TCC_OF(0),		 // Transfer Complete Code                       
		EDMA_OPT_TCCM_DEFAULT,	 // Transfer Complete Code Upper Bits (c64x only)
		EDMA_OPT_ATCINT_DEFAULT, // Alternate TCC Interrupt (c64x only)          
		EDMA_OPT_ATCC_DEFAULT,	 // Alternate Transfer Complete Code (c64x only) 
		EDMA_OPT_PDTS_DEFAULT,	 // Peripheral Device Transfer Source (c64x only)
		EDMA_OPT_PDTD_DEFAULT,	 // Peripheral Device Transfer Dest (c64x only)  
		EDMA_OPT_LINK_YES,		 // Enable link parameters                       
		EDMA_OPT_FS_NO			 // Use frame sync                               
	),							                                                 
	EDMA_SRC_OF(gBufXmtLtPing),					 // src address                                  
	EDMA_CNT_RMK(							 // Framecount/Elementcount
		EDMA_CNT_FRMCNT_OF(BUFFSIZE - 1),	 // Framecount = # of buffers -1
		EDMA_CNT_ELECNT_OF(2)				 // Elementcount = number of elements per frame
	),
	EDMA_DST_OF(0),							 // dest address
	EDMA_IDX_RMK(							 // Frame Index/Element Index
		EDMA_IDX_FRMIDX_OF(-(BUFFSIZE*2)+2), // Negative Frame Index to move us back to the previous channel
		EDMA_IDX_ELEIDX_OF(BUFFSIZE*2)		 // Positive Element Index to move us to the next channel
	),
	EDMA_RLD_RMK(							 // Element Reload needed when doing a synchronized, frame indexed transfer
		EDMA_RLD_ELERLD_OF(2),				 // Number of elements, should be the same as Element Count
		EDMA_RLD_LINK_OF(0)					 // Done in code with EDMA_link()
	)
};



/*
 *	======== initEdma ========
 */
void initEdma(void) {
	// ===== EDMA Receive Setup Code =====
	hEdmaRcv = EDMA_open(EDMA_CHA_REVT2, EDMA_OPEN_RESET);	// get hEdmaRcv handle and reset channel
	hEdmaReloadRcvPing = EDMA_allocTable(-1);				// get hEdmaReloadXmt handle
	hEdmaReloadRcvPong = EDMA_allocTable(-1);				// get hEdmaReloadXmt handle	

	gRcvTCC = EDMA_intAlloc(-1);							// get an open TCC
	gEdmaConfigRcv.opt |= EDMA_FMK(OPT,TCC,gRcvTCC);		// set TCC to gXmtChan

	gEdmaConfigRcv.src = MCBSP_getRcvAddr(hMcbspData);		// set Rcv src = McBSP0 DRR register
	
	EDMA_config(hEdmaRcv, &gEdmaConfigRcv);					// then configure that channel
	EDMA_config(hEdmaReloadRcvPing, &gEdmaConfigRcv);		// and config the reload

	gEdmaConfigRcv.dst = EDMA_DST_OF(gBufRcvLtPong);
	EDMA_config(hEdmaReloadRcvPong, &gEdmaConfigRcv);		// and config the reload	

	EDMA_link(hEdmaRcv, hEdmaReloadRcvPong);				// set up the reload addresses for all
	EDMA_link(hEdmaReloadRcvPong, hEdmaReloadRcvPing);
	EDMA_link(hEdmaReloadRcvPing, hEdmaReloadRcvPong);

	// ===== EDMA Transmit Setup Code =====
	hEdmaXmt = EDMA_open(EDMA_CHA_XEVT2, EDMA_OPEN_RESET);	// get hEdmaXmt handle and reset channel
	hEdmaReloadXmtPing = EDMA_allocTable(-1);				// get hEdmaReloadXmt handle
	hEdmaReloadXmtPong = EDMA_allocTable(-1);				// get hEdmaReloadXmt handle		 
		 
	gXmtTCC = EDMA_intAlloc(-1);							// get an open TCC
	gEdmaConfigXmt.opt |= EDMA_FMK(OPT,TCC,gXmtTCC);		// set TCC to gXmtChan

	gEdmaConfigXmt.dst = MCBSP_getXmtAddr(hMcbspData);		// set Xmt dst = McBSP0 DXR register

	EDMA_config(hEdmaXmt, &gEdmaConfigXmt);					// then configure that channel
	EDMA_config(hEdmaReloadXmtPing, &gEdmaConfigXmt);		// and config the reload	

	gEdmaConfigXmt.src = EDMA_SRC_OF(gBufXmtLtPong);
	EDMA_config(hEdmaReloadXmtPong, &gEdmaConfigXmt);		// and config the reload	

	EDMA_link(hEdmaXmt, hEdmaReloadXmtPong);				// set up the reload addresses for both
	EDMA_link(hEdmaReloadXmtPong, hEdmaReloadXmtPing);		// hEdma and hEdmaReloadXmt
	EDMA_link(hEdmaReloadXmtPing, hEdmaReloadXmtPong);		// hEdma and hEdmaReloadXmt

	// ===== Control Code =====
	EDMA_intClear(gRcvTCC);									// clear any possible spurious interrupts
	EDMA_intEnable(gRcvTCC);								// enable EDMA interrupts (CIER)
	
	EDMA_intClear(gXmtTCC);									// clear any possible spurious interrupts
	EDMA_intEnable(gXmtTCC);								// enable EDMA interrupts (CIER)

	EDMA_intHook(gRcvTCC, edmaHwi);							// hook the ISR function into the EDMA Dispatcher
	EDMA_intHook(gXmtTCC, edmaHwi);							// hook the ISR function into the EDMA Dispatcher

	EDMA_enableChannel(hEdmaXmt);							// enable both channels, waiting for sync event
	EDMA_enableChannel(hEdmaRcv);
}

/*
 *	======== edmaHwi ========
 */
void edmaHwi(int tcc)
{
    static int xmtdone = 0;
	static int rcvdone = 0;
	static int pingorpong = PING;
	short *sourceLt;
	short *sourceRt;
	short *destLt;
	short *destRt;
    
    /* Check TCC to see which transfer completed */
    if (tcc == gRcvTCC)
    {
        rcvdone = 1;
    }
    if (tcc == gXmtTCC)
    {
        xmtdone = 1;
    }
    
	/* If we have received both rcv ant xmt, process */
	if (xmtdone && rcvdone)
    {
	

		if (pingorpong == PING) {
			sourceLt = gBufRcvLtPing;
			sourceRt = gBufRcvRtPing;
			destLt = gBufXmtLtPing;
			destRt = gBufXmtRtPing;
			pingorpong = PONG;
		}
		else {
			sourceLt = gBufRcvLtPong;
			sourceRt = gBufRcvRtPong;
			destLt = gBufXmtLtPong;
			destRt = gBufXmtRtPong;
			pingorpong = PING;
		}

		if (DSK6416_DIP_get(0) == 0) {    // DIP switch 0 is on (down)	
				
		}
		
		copyData(sourceLt, destLt, BUFFSIZE);
		copyData(sourceRt, destRt, BUFFSIZE);

		rcvdone = 0;
        xmtdone = 0;
    }
}
